function [fval, row, column, checkx] = proteinlinearprogram(P)

% This program generates the weights (x1,...,xm) and (y1,...,yn) by
% converting the payoff matrix for a game theory problem into a linear programming problem. It uses
% Matlab's linprog function, which assumes that it's solving the following:
% {min(c'x) subject to Ax<=b, Hx=r, x<=u, x>=l}, where l is a vector of
% lower bounds and u is a vector of upper bounds. If a variable is
% unconstrained we can use inf to indicate an infinite bound. 
%
% Be sure that your problem is a a minimization problem!! 
%
% The program takes the following input:
% Table = the scoring table
%
% The LP is solved for the row (diversity) weight vector and the LaGrange
% multipliers are the corresponding column (conservation) weight vector.
%
% Our linear program for the row weights looks like the following:
% min v=0x1+0x2+...+0xm+v
% subject to
% p{11}x1+...+p{n1}xm-v<=0
% p{12}x1+...+p{n2}xm-v<=0
% ...
% p{1m}x1+...+p{nm}xm-v<=0
% xk>=0, k=1,2,...,m
% x1+x2+...+xm=1
%
% The program outputs the following:
% fval = the game value
% row = the row (diversity) weight vector
% column = the column (conservation) weight vector
% checkx = variable that indicates if the LP had a solution (0 if no and 1
%          if yes)
%
% Author : Brittney (Hinds) Keel
% Date created : 20 Oct 11
% Last modified : 2 Jan 15


[n, m] = size(P);

lastcolumn = -1*ones(n,1); % a column where all entries are -1
A = [P lastcolumn]; % creates the matrix A

b = zeros(n,1); % creates the vector b

% Next we'll create the matrix H:
H = ones(1,m+1); % initializes matrix H to be a matrix of ones
H(1,m+1) = 0; % makes the m+1 entry 0

r = [1]; % specifies the vector r

% Now we'll create the vector which produces the objective function.
c = zeros(m+1,1); 
c(m+1,1) = 1; % makes the m+1 coefficient 1 (the coefficient of the variable v)

l = zeros(m+1,1); % the lower bound vector (all zeros since we want probabilities)
l(m+1,1) = -inf; % makes the lower bound of the game value -inf since there is no bound

u = ones(m+1,1); % the upper bound vector (all ones to ensure probabilities)
u(m+1,1) = inf; % makes the upper bound on the game value inf since there is no bound

[x,fval,exitflag,output,lambda] = linprog(c,A,b,H,r,l,u); % uses the linprog function to
                                                          % solve the LP
                                            
                                                          
% Next determine if LP converged, and store appropriate binary value to checkx.                                                           
if numel(x) == 0 % if didn't converge set all variables to zero
    checkx = 0; 
    row = 0;
    column = 0;
    fval = 0;
else % if did converge                                                        
    for i = 1:m
        row(i) = x(i); % the row weight vector (from the LP solution) 
    end

    for j = 1:n
        column(j) = lambda.ineqlin(j); % the column weight vector (the LaGrange multipliers
                                       % or shadow price of the LP solution)
    end
    checkx = 1; % indicates solution was found
end